<?php
/**
 * PPOM Inputs Meta Manager Class
 *
 * It control the inputs meta data. It show all the global inputs settings.
 *
 * @version  1.0
 */

/* 
**========== Block direct access =========== 
*/
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class PPOM_InputManager {

	/**
	 * Return all ppom inputs meta data
	 *
	 * @var array
	 */
	public static $input_meta;

	/**
	 * Return input type
	 *
	 * @var string
	 */
	public $input_type;

	/* ======= Class Construct ======== */
	function __construct( $input_meta, $input_type ) {

		self::$input_meta = $input_meta;

		$this->input_type = $input_type;
	}


	/**
	 * Field Title
	 *
	 * @hook ppom_{$field_type}_input_meta_title
	 */
	function title() {

		$title = isset( self::$input_meta['title'] ) ? stripslashes( self::$input_meta['title'] ) : '';

		$title = ppom_wpml_translate( $title, 'PPOM' );

		return apply_filters( 'ppom_input_meta_title', $title, self::$input_meta );
	}


	/**
	 * Field Desciption
	 *
	 * @hook ppom_{$field_type}_input_meta_desc
	 * @hook ppom_description_content
	 */
	function desc() {

		$desc = isset( self::$input_meta['description'] ) ? stripslashes( self::$input_meta['description'] ) : '';

		$desc = ppom_wpml_translate( $desc, 'PPOM' );

		// old Filter
		$desc = apply_filters( 'ppom_description_content', $desc, self::$input_meta );

		return apply_filters( 'ppom_input_meta_desc', $desc, self::$input_meta );
	}


	/**
	 * Field Required
	 *
	 * @hook ppom_{$field_type}_input_meta_required
	 */
	function required() {

		$required = isset( self::$input_meta['required'] ) ? self::$input_meta['required'] : '';

		return apply_filters( 'ppom_input_meta_required', $required, self::$input_meta );
	}


	/**
	 * Field Desc Tooltip
	 */
	function enable_tooltip() {

		$required = isset( self::$input_meta['desc_tooltip'] ) ? self::$input_meta['desc_tooltip'] : '';

		return $required;
	}


	/**
	 * Field dataname (Field Unique ID)
	 *
	 * @hook ppom_{$field_type}_input_meta_data_name
	 */
	function data_name() {

		$data_name = isset( self::$input_meta['data_name'] ) ? sanitize_key( self::$input_meta['data_name'] ) : $this->title();

		return apply_filters( 'ppom_input_meta_data_name', $data_name, self::$input_meta );
	}


	/**
	 * Field Placeholder
	 *
	 * @hook ppom_{$field_type}_input_meta_placeholder
	 */
	function placeholder() {

		$placeholder = isset( self::$input_meta['placeholder'] ) ? stripslashes( self::$input_meta['placeholder'] ) : '';

		$placeholder = ppom_wpml_translate( $placeholder, 'PPOM' );

		return apply_filters( 'ppom_input_meta_placeholder', $placeholder, self::$input_meta );
	}


	/**
	 * Field Error Message
	 *
	 * @hook ppom_{$field_type}_input_meta_error_msg
	 */
	function error_msg() {

		$error_msg = isset( self::$input_meta['error_message'] ) ? self::$input_meta['error_message'] : '';

		$error_msg = ppom_wpml_translate( $error_msg, 'PPOM' );

		return apply_filters( 'ppom_input_meta_error_msg', $error_msg, self::$input_meta );
	}


	/**
	 * Field Label
	 *
	 * Show Asterisk If Require On
	 *
	 * Show Description If Not Null
	 *
	 * @hook ppom_field_description
	 * @hook ppom_{$field_type}_input_meta_label_html
	 */
	function field_label( $tooltip = true, $desc = true, $asterisk = true ) {

		$asterisk_symbol = ( ! empty( $this->required() ) && $this->title() != '' ) ? '<span class="show_required"> *</span>' : '';

		$show_desc = ( ! empty( $this->desc() ) ) ? '<span class="show_description ppom-input-desc">' . $this->desc() . '</span>' : '';

		if ( $desc ) {
			$show_desc = apply_filters( 'ppom_field_description', $show_desc, self::$input_meta );
		}

		$field_label = $this->title();
		// $field_label = $this->title() . $asterisk_symbol . $show_desc;

		if ( $asterisk ) {
			$field_label = $field_label . $asterisk_symbol;
		}

		if ( $tooltip ) {
			$field_label = $field_label . $show_desc;
		}

		return apply_filters( 'ppom_input_meta_label_html', $field_label, self::$input_meta );
	}


	/**
	 * Field Desciption With Tooltip
	 *
	 * @hook ppom_input_meta_tooltip_desc
	 */
	function tooltip() {

		$show_desc = ( ! empty( $this->desc() ) ) ? '<span class="show_description ppom-input-desc">' . $this->desc() . '</span>' : '';
		$show_desc = apply_filters( 'ppom_field_description', $show_desc, self::$input_meta );

		return apply_filters( 'ppom_input_meta_tooltip_desc', $show_desc, self::$input_meta );
	}


	/**
	 * Field Multiple Options
	 *
	 * Checkbox|Radio|Select|Image|Pallete
	 *
	 * @hook ppom_{$field_type}_input_meta_multi_options
	 */
	function options() {

		$options = isset( self::$input_meta['options'] ) ? self::$input_meta['options'] : array();

		if ( is_array( $options ) ) {
			$options = array_map( 'ppom_translation_options', $options );
		}

		return apply_filters( 'ppom_input_meta_multi_options', $options, self::$input_meta );
	}


	/**
	 * Images Options
	 *
	 * @hook ppom_{$field_type}_input_meta_images
	 */
	function images() {

		$images = isset( self::$input_meta['images'] ) ? self::$input_meta['images'] : array();

		return apply_filters( 'ppom_input_meta_images', $images, self::$input_meta );
	}


	/**
	 * Audio/Video Options
	 *
	 * @hook ppom_{$field_type}_input_meta_audio
	 */
	function audio_video() {

		$audios = isset( self::$input_meta['audio'] ) ? self::$input_meta['audio'] : array();

		return apply_filters( 'ppom_input_meta_audio', $audios, self::$input_meta );
	}


	/*
	===================================
		Wrapper Classes Section
	===================================*/


	/**
	 * Field inner Wrapper Classes
	 *
	 * @hook ppom_input_wrapper_class
	 */
	function field_inner_wrapper_classes() {

		$classes         = [ 'form-group' ];
		$wrapper_classes = implode( ' ', $classes );

		// return apply_filters_deprecated( 'ppom_input_wrapper_class', array( $wrapper_classes, self::$input_meta ), '21.3', 'ppom_input_wrapper_classes' );
		return apply_filters( 'ppom_input_wrapper_class', $wrapper_classes, self::$input_meta );
	}


	/**
	 * Field Label Classes
	 *
	 * @hook ppom_{$this->input_type}_input_label_classes
	 */
	function label_classes() {

		$classes = [ 'form-control-label' ];

		$label_classes = apply_filters( 'ppom_input_label_classes', $classes, self::$input_meta );

		$label_classes = implode( ' ', $label_classes );

		return $label_classes;
	}


	/**
	 * Field Classes Array
	 *
	 * @hook ppom_{$this->input_type}_input_meta_classes
	 * @hook ppom_input_classes
	 */
	function input_classes_array() {

		$classes = isset( self::$input_meta['class'] ) ? explode( ',', self::$input_meta['class'] ) : array();

		if ( ! empty( $classes ) ) {
			$classes[] = 'form-control';
		} else {
			$classes = array( 'form-control' );
		}

		if ( $this->input_type == 'color' ) {
			$classes[] = 'text';
		} else {
			$classes[] = $this->input_type;
		}

		$classes[] = 'ppom-input';
		// TODO: re-check again
		// if ($this->input_type != 'image') {
		// }

		if ( $this->required() ) {
			$classes[] = 'ppom-required';
		}

		if ( ( $this->input_type == 'radio' && ( $key = array_search( 'form-control', $classes ) ) !== false ) ||
			 $this->input_type == 'checkbox' && ( $key = array_search( 'form-control', $classes ) ) !== false ) {
			unset( $classes[ $key ] );
			$classes[] = 'ppom-check-input';
		}

		if ( $this->input_type == 'select' && ( $key = array_search( 'form-control', $classes ) ) !== false ) {
			unset( $classes[ $key ] );
			$classes[] = 'form-select';
		}

		$classes = array_filter( apply_filters( 'ppom_input_meta_classes', $classes, self::$input_meta ) );

		return $classes;
	}


	/**
	 * Field Classes
	 *
	 * @hook ppom_{$this->input_type}_input_meta_classes
	 * @hook ppom_input_classes
	 */
	function input_classes() {

		$classes = $this->input_classes_array();

		// $input_classes = apply_filters_deprecated( 'ppom_input_classes', array( $classes, self::$input_meta ), '21.3', 'ppom_form_input_classes' );
		$input_classes = apply_filters( 'ppom_input_classes', $classes, self::$input_meta );

		$input_classes = implode( ' ', $input_classes );

		return $input_classes;
	}


	/**
	 * Radio Input label classes
	 *
	 * @hook ppom_radio_input_label_classes
	 */
	function radio_label_classes() {

		$classes = [ 'form-check-label' ];

		$label_class = apply_filters( 'ppom_radio_input_label_classes', $classes, self::$input_meta );

		$label_class = implode( ' ', $label_class );

		return $label_class;
	}


	/**
	 * Checkbox Input label classes
	 *
	 * @hook ppom_checkbox_input_label_classes
	 */
	function checkbox_label_classes() {

		$classes = [ 'form-check-label' ];

		$label_class = apply_filters( 'ppom_checkbox_input_label_classes', $classes, self::$input_meta );

		$label_class = implode( ' ', $label_class );

		return $label_class;
	}


	/**
	 * Generate Field Attribute Name Key
	 *
	 * @hook ppom_{$this->input_type}_input_name_attr
	 */
	function form_name() {

		$form_name = 'ppom[fields][' . esc_attr( $this->data_name() ) . ']';
		if ( $this->input_type == 'checkbox' ) {
			$form_name .= '[]';
		}

		return apply_filters( 'ppom_input_name_attr', $form_name, self::$input_meta );
	}


	/**
	 * Get input meta value by key
	 *
	 * @hook ppom_{$this->input_type}_input_meta_value_by_key
	 */
	function get_meta_value( $key, $default = null ) {

		$value = ! is_null( $default ) ? $default : '';
		if ( isset( self::$input_meta[ $key ] ) && self::$input_meta[ $key ] != '' ) {
			$value = self::$input_meta[ $key ];
		}

		$value = apply_filters( "ppom_{$this->input_type}_input_meta_value_by_key", $value, $key, self::$input_meta );

		return apply_filters( 'ppom_field_meta_value', $value, $key, self::$input_meta );
	}
}
