<?php
/**
 * The dashboard-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the dashboard-specific stylesheet and JavaScript.
 *
 * @package    Equal_Height_Columns
 * @subpackage Equal_Height_Columns/admin
 * @author     MIGHTYminnow, Mickey Kay, Braad Martin
 */
class Equal_Height_Columns_Admin {

	/**
	 * The ID of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_name    The ID of this plugin.
	 */
	private $plugin_name;

	/**
	 * The display name of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $plugin_display_name    The public name of this plugin.
	 */
	private $plugin_display_name;

	/**
	 * The version of this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $version    The current version of this plugin.
	 */
	private $version;

	/**
	 * Options for this plugin.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      array    $options    The options stored for this plugin.
	 */
	private $options;

	/**
	 * Slug for main options group.
	 *
	 * @since    1.0.0
	 * @access   private
	 * @var      string    $this->options_group_slug    Slug for the main options group.
	 */
	private $options_group_slug;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since    1.0.0
	 * @var      string    $plugin_name            The name of this plugin.
	 * @var      string    $plugin_display_name    The public name of this plugin.
	 * @var      string    $version                The version of this plugin.
	 */
	public function __construct( $plugin_name, $plugin_display_name, $version ) {

		$this->plugin_name = $plugin_name;
		$this->plugin_display_name = $plugin_display_name;
		$this->version = $version;
		$this->options_group_slug = 'element-groups';

	}

	/**
	 * Register the stylesheets for the Dashboard.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_styles() {

		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'css/equal-height-columns-admin.css', array(), $this->version, 'all' );

	}

	/**
	 * Register the JavaScript for the dashboard.
	 *
	 * @since    1.0.0
	 */
	public function enqueue_scripts( $hook = '' ) {

		if ( ( 'settings_page_' . $this->plugin_name ) == $hook ) {
			wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'js/equal-height-columns-admin.js', array( 'jquery' ), $this->version, false );
		}

	}

	/**
	 * Add options page to admin menu.
	 *
	 * @since    1.0.0
	 */
	public function add_options_page() {

		// Get the options stored for this plugin.
		$this->options = get_option( $this->plugin_name );

		add_options_page(
            $this->plugin_display_name, // Page title
            $this->plugin_display_name, // Menu title
            'manage_options', // Capability
            $this->plugin_name, // Menu slug
            array( $this, 'output_options_page' )
        );

	}

	/**
	 * Output the contents of the options page.
	 *
	 * @since    1.0.0
	 */
	public function output_options_page() {
		?>
		<div class="wrap mm-settings">
            <?php screen_icon(); ?>
            <h2><?php echo $this->plugin_display_name; ?></h2>
            <div id="poststuff">
                <div id="post-body" class="columns-2">
                    <div id="post-body-content">
                        <form method="post" action="options.php">
                        <?php
                            settings_fields( $this->plugin_name );
			                do_settings_sections( $this->plugin_name );
			                $this->output_add_button();
			                submit_button();
                        ?>
                        </form>
                    </div>
                    <div id="postbox-container-1" class="postbox-container">
                        <div class="postbox">
                            <h3>MIGHTYminnow</h3>
                            <div class="inside">
                            	<?php
                            		// Form MIGHTYminnow link with plugin name for tracking
									$plugin_slug = 'equal-height-columns';
									$link = '//mightyminnow.com/plugin-landing-page?utm_source=' . $plugin_slug . '&utm_medium=plugin&utm_campaign=WordPress%20Plugins';
								?>
                                <a class="logo-wrap" href="<?php echo $link; ?>" target="_blank">
                                	<svg class="fish" version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px"
                                	viewBox="0 0 44.072 18.075" enable-background="new 0 0 44.072 18.075" xml:space="preserve">
		                            	<path fill="#231F20" d="M15.882,12.234c4.078,1.897,9.253,2.505,14.585,1.37c5.864-1.246,10.702-4.34,13.606-8.223
		                            	c-4.169-2.143-9.615-2.876-15.231-1.684c-5.48,1.165-10.065,3.943-13.011,7.473C11.185,8.431,7.171,4.496,3.446,0
		                            	C4.552,6.494,3.246,12.485,0,18.075C4.912,15.579,9.906,13.208,15.882,12.234z"/>
		                            </svg>
                               		<svg class="letters" version="1.1" id="Layer_1" xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" x="0px" y="0px" viewBox="0 0 165.906 106.259" enable-background="new 0 0 165.906 106.259" xml:space="preserve">
                                		<g>
                                			<g class="mm">
                                				<path fill="#231F20" d="M80.15,0v82.268H64.05V56.529c0-17.275,0.588-29.147,0.47-29.147h-0.117c-0.117,0-5.876,10.107-7.286,12.459L39.995,65.57L23.034,39.842c-1.528-2.352-7.286-12.459-7.404-12.459h-0.117c-0.117,0,0.588,11.871,0.588,29.147v25.738H0V0h15.395l17.04,28.207c6.347,10.459,7.404,13.045,7.522,13.045h0.118c0.117,0,1.175-2.586,7.521-13.045L64.755,0H80.15z"/>
                                				<path fill="#231F20" d="M91.856,33.488h6.276v8.43c2.185-3.109,4.428-5.381,6.732-6.814c3.173-1.914,6.509-2.871,10.01-2.871c2.364,0,4.607,0.465,6.732,1.391c2.125,0.928,3.86,2.16,5.207,3.699c1.347,1.539,2.514,3.744,3.501,6.613c2.087-3.826,4.695-6.732,7.827-8.721c3.131-1.988,6.501-2.982,10.108-2.982c3.369,0,6.344,0.854,8.923,2.557c2.579,1.703,4.495,4.08,5.747,7.129c1.253,3.049,1.879,7.621,1.879,13.719v26.631h-6.411V55.637c0-5.23-0.373-8.824-1.117-10.783c-0.745-1.957-2.019-3.533-3.82-4.73c-1.803-1.193-3.955-1.793-6.457-1.793c-3.039,0-5.816,0.896-8.334,2.691c-2.518,1.793-4.357,4.184-5.519,7.172c-1.162,2.99-1.742,7.981-1.742,14.975v19.1h-6.277V57.295c0-5.889-0.366-9.885-1.098-11.992c-0.733-2.107-2.011-3.797-3.834-5.066c-1.823-1.27-3.989-1.906-6.5-1.906c-2.899,0-5.612,0.875-8.138,2.623c-2.526,1.748-4.394,4.102-5.604,7.061c-1.21,2.959-1.815,7.488-1.815,13.586v20.668h-6.276V33.488z"/>
                                			</g>
                                			<g class="subtext-1">
                                				<path fill="#231F20" d="M0.361,93.484h1.195l2.769,6.557l2.882-6.557h0.211l2.9,6.557l2.822-6.557h1.208l-3.923,9.031H10.21l-2.888-6.467l-2.887,6.467H4.22L0.361,93.484z"/>
                                				<path fill="#231F20" d="M23.279,99.519l0.979,0.515c-0.321,0.631-0.692,1.141-1.113,1.527s-0.895,0.682-1.421,0.885c-0.526,0.201-1.122,0.303-1.786,0.303c-1.474,0-2.626-0.483-3.456-1.449c-0.831-0.965-1.246-2.057-1.246-3.274c0-1.146,0.352-2.167,1.055-3.063c0.892-1.14,2.085-1.709,3.581-1.709c1.54,0,2.77,0.584,3.688,1.751c0.653,0.825,0.986,1.854,0.997,3.088h-8.126c0.021,1.049,0.356,1.909,1.005,2.579c0.648,0.672,1.449,1.007,2.402,1.007c0.46,0,0.907-0.08,1.342-0.239c0.435-0.16,0.805-0.373,1.109-0.637C22.595,100.535,22.925,100.108,23.279,99.519z M23.279,97.086c-0.155-0.619-0.381-1.115-0.678-1.485c-0.296-0.37-0.688-0.669-1.176-0.896s-1-0.341-1.537-0.341c-0.887,0-1.648,0.285-2.286,0.855c-0.465,0.415-0.817,1.037-1.056,1.867H23.279z"/>
                                				<path fill="#231F20" d="M26.765,102.516V89.998h1.162v5.038c0.487-0.598,1.031-1.044,1.634-1.341c0.603-0.296,1.264-0.443,1.982-0.443c1.276,0,2.367,0.463,3.271,1.391c0.903,0.926,1.355,2.051,1.355,3.373c0,1.307-0.456,2.422-1.368,3.346s-2.01,1.387-3.292,1.387c-0.735,0-1.401-0.158-1.999-0.474c-0.597-0.315-1.125-0.788-1.584-1.419v1.66H26.765z M31.413,101.627c0.646,0,1.243-0.159,1.789-0.477c0.547-0.318,0.981-0.766,1.305-1.342c0.323-0.575,0.484-1.184,0.484-1.826c0-0.642-0.163-1.253-0.488-1.834c-0.326-0.581-0.764-1.033-1.312-1.357c-0.55-0.324-1.137-0.485-1.76-0.485c-0.636,0-1.238,0.161-1.811,0.485c-0.571,0.324-1.01,0.76-1.316,1.308s-0.459,1.165-0.459,1.851c0,1.047,0.343,1.921,1.03,2.623C29.562,101.275,30.408,101.627,31.413,101.627z"/>
                                				<path fill="#231F20" d="M47.466,94.472l-0.748,0.772c-0.621-0.604-1.229-0.905-1.822-0.905c-0.377,0-0.7,0.124-0.97,0.374c-0.269,0.248-0.403,0.539-0.403,0.871c0,0.293,0.111,0.572,0.333,0.838c0.222,0.271,0.688,0.59,1.397,0.955c0.865,0.448,1.452,0.88,1.764,1.295c0.305,0.42,0.457,0.894,0.457,1.42c0,0.741-0.261,1.369-0.781,1.884c-0.521,0.515-1.173,0.772-1.954,0.772c-0.521,0-1.019-0.113-1.492-0.341s-0.866-0.54-1.177-0.938l0.73-0.83c0.594,0.67,1.224,1.005,1.89,1.005c0.466,0,0.862-0.149,1.189-0.448s0.491-0.65,0.491-1.055c0-0.332-0.108-0.627-0.325-0.888c-0.216-0.255-0.704-0.575-1.464-0.963c-0.815-0.421-1.37-0.835-1.664-1.245s-0.44-0.877-0.44-1.402c0-0.687,0.234-1.257,0.702-1.711c0.469-0.453,1.061-0.68,1.775-0.68C45.786,93.252,46.623,93.658,47.466,94.472z"/>
                                				<path fill="#231F20" d="M50.503,90.131h1.162v3.354h1.843v1.004h-1.843v8.027h-1.162v-8.027h-1.585v-1.004h1.585V90.131z"/>
                                				<path fill="#231F20" d="M55.067,93.484h1.162v4.217c0,1.029,0.056,1.737,0.166,2.125c0.167,0.553,0.483,0.99,0.952,1.311c0.468,0.321,1.026,0.482,1.675,0.482s1.198-0.156,1.65-0.469c0.451-0.313,0.764-0.725,0.936-1.233c0.116-0.349,0.175-1.087,0.175-2.216v-4.217h1.187v4.432c0,1.246-0.146,2.184-0.436,2.814c-0.291,0.631-0.728,1.125-1.312,1.482c-0.584,0.355-1.316,0.535-2.195,0.535c-0.881,0-1.615-0.18-2.204-0.535c-0.59-0.357-1.029-0.855-1.32-1.494c-0.29-0.641-0.436-1.602-0.436-2.885V93.484z"/>
                                				<path fill="#231F20" d="M74.532,89.998v12.518h-1.146v-1.553c-0.487,0.592-1.033,1.037-1.64,1.336c-0.605,0.299-1.269,0.449-1.987,0.449c-1.279,0-2.37-0.464-3.275-1.391c-0.904-0.928-1.357-2.055-1.357-3.383c0-1.301,0.457-2.412,1.37-3.337s2.012-1.386,3.296-1.386c0.741,0,1.412,0.157,2.013,0.473s1.127,0.789,1.581,1.42v-5.146H74.532z M69.888,94.372c-0.646,0-1.243,0.159-1.791,0.477c-0.547,0.317-0.982,0.765-1.306,1.339c-0.323,0.575-0.484,1.184-0.484,1.824c0,0.637,0.162,1.244,0.488,1.824s0.765,1.032,1.314,1.355c0.55,0.324,1.14,0.485,1.771,0.485c0.636,0,1.238-0.16,1.808-0.481c0.569-0.32,1.007-0.754,1.313-1.301s0.46-1.164,0.46-1.85c0-1.045-0.344-1.918-1.032-2.619C71.741,94.723,70.894,94.372,69.888,94.372z"/>
                                				<path fill="#231F20" d="M77.531,89.766c0.265,0,0.49,0.094,0.678,0.281c0.188,0.188,0.281,0.416,0.281,0.682c0,0.26-0.094,0.484-0.281,0.672s-0.413,0.282-0.678,0.282c-0.259,0-0.482-0.095-0.669-0.282c-0.188-0.188-0.281-0.412-0.281-0.672c0-0.266,0.094-0.493,0.281-0.682C77.049,89.859,77.272,89.766,77.531,89.766z M76.955,93.484h1.162v9.031h-1.162V93.484z"/>
                                				<path fill="#231F20" d="M84.774,93.252c1.391,0,2.544,0.503,3.459,1.51c0.831,0.919,1.247,2.006,1.247,3.263c0,1.262-0.439,2.364-1.318,3.308c-0.878,0.943-2.008,1.416-3.388,1.416c-1.387,0-2.519-0.473-3.397-1.416s-1.317-2.046-1.317-3.308c0-1.251,0.415-2.335,1.247-3.254C82.221,93.758,83.377,93.252,84.774,93.252z M84.77,94.389c-0.966,0-1.796,0.357-2.49,1.07c-0.694,0.715-1.042,1.578-1.042,2.59c0,0.653,0.159,1.262,0.476,1.826s0.744,1,1.282,1.309c0.539,0.307,1.131,0.46,1.774,0.46c0.645,0,1.235-0.153,1.774-0.46c0.539-0.309,0.967-0.744,1.283-1.309s0.475-1.173,0.475-1.826c0-1.012-0.349-1.875-1.046-2.59C86.56,94.746,85.73,94.389,84.77,94.389z"/>
                                				<path fill="#231F20" d="M105.698,98l0.827,0.875c-0.531,0.471-1.035,0.877-1.513,1.219c0.332,0.306,0.766,0.726,1.3,1.26c0.422,0.42,0.784,0.807,1.087,1.162h-1.702l-1.619-1.67c-1.018,0.82-1.844,1.354-2.48,1.604c-0.637,0.248-1.309,0.373-2.017,0.373c-0.941,0-1.696-0.27-2.267-0.81s-0.854-1.214-0.854-2.022c0-0.608,0.191-1.219,0.576-1.83c0.384-0.611,1.168-1.386,2.352-2.321c-0.669-0.808-1.099-1.403-1.289-1.788s-0.286-0.75-0.286-1.1c0-0.607,0.235-1.111,0.706-1.51c0.542-0.465,1.217-0.697,2.025-0.697c0.509,0,0.974,0.104,1.395,0.308c0.42,0.204,0.741,0.479,0.963,0.825c0.221,0.347,0.332,0.713,0.332,1.101c0,0.409-0.145,0.845-0.433,1.308c-0.288,0.462-0.856,1.053-1.704,1.771L102.948,98l1.213,1.229C104.812,98.75,105.325,98.341,105.698,98z M100.192,96.76c-1.037,0.781-1.717,1.402-2.038,1.865c-0.322,0.462-0.483,0.89-0.483,1.283c0,0.465,0.197,0.876,0.591,1.232c0.394,0.357,0.876,0.536,1.447,0.536c0.433,0,0.856-0.089,1.272-0.266c0.654-0.282,1.404-0.761,2.25-1.438l-1.927-1.984C100.973,97.641,100.603,97.23,100.192,96.76z M100.342,95.145c0.593-0.46,1.039-0.91,1.338-1.354c0.211-0.315,0.316-0.595,0.316-0.838c0-0.283-0.138-0.526-0.412-0.73c-0.274-0.205-0.64-0.308-1.095-0.308c-0.433,0-0.785,0.104-1.057,0.315c-0.272,0.211-0.408,0.459-0.408,0.747c0,0.222,0.056,0.429,0.167,0.622C99.463,94.07,99.847,94.585,100.342,95.145z"/>
                                				<path fill="#231F20" d="M118.314,94.472l-0.748,0.772c-0.621-0.604-1.229-0.905-1.822-0.905c-0.377,0-0.7,0.124-0.97,0.374c-0.269,0.248-0.403,0.539-0.403,0.871c0,0.293,0.111,0.572,0.333,0.838c0.222,0.271,0.688,0.59,1.397,0.955c0.865,0.448,1.452,0.88,1.764,1.295c0.305,0.42,0.457,0.894,0.457,1.42c0,0.741-0.261,1.369-0.781,1.884c-0.521,0.515-1.173,0.772-1.954,0.772c-0.521,0-1.019-0.113-1.492-0.341s-0.866-0.54-1.177-0.938l0.73-0.83c0.594,0.67,1.224,1.005,1.89,1.005c0.466,0,0.862-0.149,1.189-0.448s0.491-0.65,0.491-1.055c0-0.332-0.108-0.627-0.325-0.888c-0.216-0.255-0.704-0.575-1.464-0.963c-0.815-0.421-1.37-0.835-1.664-1.245s-0.44-0.877-0.44-1.402c0-0.687,0.234-1.257,0.702-1.711c0.469-0.453,1.061-0.68,1.775-0.68C116.635,93.252,117.472,93.658,118.314,94.472z"/>
                                				<path fill="#231F20" d="M129.104,95.359l-0.922,0.573c-0.796-1.058-1.882-1.585-3.258-1.585c-1.101,0-2.014,0.354-2.74,1.061c-0.727,0.708-1.09,1.567-1.09,2.578c0,0.658,0.167,1.277,0.501,1.857c0.334,0.581,0.793,1.031,1.376,1.352c0.584,0.321,1.237,0.481,1.961,0.481c1.327,0,2.41-0.528,3.25-1.585l0.922,0.605c-0.432,0.65-1.012,1.154-1.739,1.513c-0.728,0.358-1.556,0.538-2.485,0.538c-1.428,0-2.612-0.454-3.553-1.361c-0.941-0.908-1.412-2.012-1.412-3.312c0-0.875,0.221-1.687,0.66-2.436c0.44-0.75,1.045-1.336,1.814-1.756c0.769-0.421,1.629-0.631,2.581-0.631c0.598,0,1.175,0.091,1.73,0.273c0.557,0.183,1.028,0.42,1.415,0.714C128.504,94.533,128.833,94.906,129.104,95.359z"/>
                                				<path fill="#231F20" d="M131.303,89.998h1.162v5.104c0.471-0.62,0.988-1.083,1.553-1.39c0.564-0.309,1.176-0.461,1.834-0.461c0.675,0,1.274,0.171,1.797,0.514c0.523,0.344,0.909,0.805,1.158,1.383s0.373,1.484,0.373,2.719v4.648h-1.162v-4.309c0-1.041-0.041-1.734-0.124-2.084c-0.144-0.598-0.405-1.047-0.784-1.348c-0.379-0.303-0.876-0.453-1.49-0.453c-0.703,0-1.332,0.232-1.889,0.697c-0.556,0.465-0.923,1.041-1.1,1.727c-0.11,0.443-0.166,1.262-0.166,2.457v3.312h-1.162V89.998z"/>
                                				<path fill="#231F20" d="M145.912,93.252c1.391,0,2.544,0.503,3.459,1.51c0.831,0.919,1.247,2.006,1.247,3.263c0,1.262-0.439,2.364-1.318,3.308c-0.878,0.943-2.008,1.416-3.388,1.416c-1.387,0-2.519-0.473-3.397-1.416s-1.317-2.046-1.317-3.308c0-1.251,0.415-2.335,1.247-3.254C143.358,93.758,144.515,93.252,145.912,93.252z M145.907,94.389c-0.966,0-1.796,0.357-2.49,1.07c-0.694,0.715-1.042,1.578-1.042,2.59c0,0.653,0.159,1.262,0.476,1.826s0.744,1,1.282,1.309c0.539,0.307,1.131,0.46,1.774,0.46c0.645,0,1.235-0.153,1.774-0.46c0.539-0.309,0.967-0.744,1.283-1.309s0.475-1.173,0.475-1.826c0-1.012-0.349-1.875-1.046-2.59C147.697,94.746,146.868,94.389,145.907,94.389z"/>
                                				<path fill="#231F20" d="M157.01,93.252c1.391,0,2.544,0.503,3.459,1.51c0.831,0.919,1.247,2.006,1.247,3.263c0,1.262-0.439,2.364-1.318,3.308c-0.878,0.943-2.008,1.416-3.388,1.416c-1.387,0-2.519-0.473-3.397-1.416s-1.317-2.046-1.317-3.308c0-1.251,0.415-2.335,1.247-3.254C154.456,93.758,155.612,93.252,157.01,93.252z M157.005,94.389c-0.966,0-1.796,0.357-2.49,1.07c-0.694,0.715-1.042,1.578-1.042,2.59c0,0.653,0.159,1.262,0.476,1.826s0.744,1,1.282,1.309c0.539,0.307,1.131,0.46,1.774,0.46c0.645,0,1.235-0.153,1.774-0.46c0.539-0.309,0.967-0.744,1.283-1.309s0.475-1.173,0.475-1.826c0-1.012-0.349-1.875-1.046-2.59C158.795,94.746,157.966,94.389,157.005,94.389z"/>
                                				<path fill="#231F20" d="M163.657,89.998h1.162v12.518h-1.162V89.998z"/>
                                			</g>
                                			<g class="subtext-2">
                                				<path fill="#231F20" d="M0.361,93.484h1.195l2.769,6.557l2.882-6.557h0.211l2.9,6.557l2.822-6.557h1.208l-3.923,9.031H10.21l-2.888-6.467l-2.887,6.467H4.22L0.361,93.484z"/>
                                				<path fill="#231F20" d="M23.279,99.519l0.979,0.515c-0.321,0.631-0.692,1.141-1.113,1.527s-0.895,0.682-1.421,0.885c-0.526,0.201-1.122,0.303-1.786,0.303c-1.474,0-2.626-0.483-3.456-1.449c-0.831-0.965-1.246-2.057-1.246-3.274c0-1.146,0.352-2.167,1.055-3.063c0.892-1.14,2.085-1.709,3.581-1.709c1.54,0,2.77,0.584,3.688,1.751c0.653,0.825,0.986,1.854,0.997,3.088h-8.126c0.021,1.049,0.356,1.909,1.005,2.579c0.648,0.672,1.449,1.007,2.402,1.007c0.46,0,0.907-0.08,1.342-0.239c0.435-0.16,0.805-0.373,1.109-0.637C22.595,100.535,22.925,100.108,23.279,99.519z M23.279,97.086c-0.155-0.619-0.381-1.115-0.678-1.485c-0.296-0.37-0.688-0.669-1.176-0.896s-1-0.341-1.537-0.341c-0.887,0-1.648,0.285-2.286,0.855c-0.465,0.415-0.817,1.037-1.056,1.867H23.279z"/>
                                				<path fill="#231F20" d="M26.765,102.516V89.998h1.162v5.038c0.487-0.598,1.031-1.044,1.634-1.341c0.603-0.296,1.264-0.443,1.982-0.443c1.276,0,2.367,0.463,3.271,1.391c0.903,0.926,1.355,2.051,1.355,3.373c0,1.307-0.456,2.422-1.368,3.346s-2.01,1.387-3.292,1.387c-0.735,0-1.401-0.158-1.999-0.474c-0.597-0.315-1.125-0.788-1.584-1.419v1.66H26.765z M31.413,101.627c0.646,0,1.243-0.159,1.789-0.477c0.547-0.318,0.981-0.766,1.305-1.342c0.323-0.575,0.484-1.184,0.484-1.826c0-0.642-0.163-1.253-0.488-1.834c-0.326-0.581-0.764-1.033-1.312-1.357c-0.55-0.324-1.137-0.485-1.76-0.485c-0.636,0-1.238,0.161-1.811,0.485c-0.571,0.324-1.01,0.76-1.316,1.308s-0.459,1.165-0.459,1.851c0,1.047,0.343,1.921,1.03,2.623C29.562,101.275,30.408,101.627,31.413,101.627z"/>
                                				<path fill="#231F20" d="M47.466,94.472l-0.748,0.772c-0.621-0.604-1.229-0.905-1.822-0.905c-0.377,0-0.7,0.124-0.97,0.374c-0.269,0.248-0.403,0.539-0.403,0.871c0,0.293,0.111,0.572,0.333,0.838c0.222,0.271,0.688,0.59,1.397,0.955c0.865,0.448,1.452,0.88,1.764,1.295c0.305,0.42,0.457,0.894,0.457,1.42c0,0.741-0.261,1.369-0.781,1.884c-0.521,0.515-1.173,0.772-1.954,0.772c-0.521,0-1.019-0.113-1.492-0.341s-0.866-0.54-1.177-0.938l0.73-0.83c0.594,0.67,1.224,1.005,1.89,1.005c0.466,0,0.862-0.149,1.189-0.448s0.491-0.65,0.491-1.055c0-0.332-0.108-0.627-0.325-0.888c-0.216-0.255-0.704-0.575-1.464-0.963c-0.815-0.421-1.37-0.835-1.664-1.245s-0.44-0.877-0.44-1.402c0-0.687,0.234-1.257,0.702-1.711c0.469-0.453,1.061-0.68,1.775-0.68C45.786,93.252,46.623,93.658,47.466,94.472z"/>
                                				<path fill="#231F20" d="M50.503,90.131h1.162v3.354h1.843v1.004h-1.843v8.027h-1.162v-8.027h-1.585v-1.004h1.585V90.131z"/>
                                				<path fill="#231F20" d="M55.067,93.484h1.162v4.217c0,1.029,0.056,1.737,0.166,2.125c0.167,0.553,0.483,0.99,0.952,1.311c0.468,0.321,1.026,0.482,1.675,0.482s1.198-0.156,1.65-0.469c0.451-0.313,0.764-0.725,0.936-1.233c0.116-0.349,0.175-1.087,0.175-2.216v-4.217h1.187v4.432c0,1.246-0.146,2.184-0.436,2.814c-0.291,0.631-0.728,1.125-1.312,1.482c-0.584,0.355-1.316,0.535-2.195,0.535c-0.881,0-1.615-0.18-2.204-0.535c-0.59-0.357-1.029-0.855-1.32-1.494c-0.29-0.641-0.436-1.602-0.436-2.885V93.484z"/>
                                				<path fill="#231F20" d="M74.532,89.998v12.518h-1.146v-1.553c-0.487,0.592-1.033,1.037-1.64,1.336c-0.605,0.299-1.269,0.449-1.987,0.449c-1.279,0-2.37-0.464-3.275-1.391c-0.904-0.928-1.357-2.055-1.357-3.383c0-1.301,0.457-2.412,1.37-3.337s2.012-1.386,3.296-1.386c0.741,0,1.412,0.157,2.013,0.473s1.127,0.789,1.581,1.42v-5.146H74.532z M69.888,94.372c-0.646,0-1.243,0.159-1.791,0.477c-0.547,0.317-0.982,0.765-1.306,1.339c-0.323,0.575-0.484,1.184-0.484,1.824c0,0.637,0.162,1.244,0.488,1.824s0.765,1.032,1.314,1.355c0.55,0.324,1.14,0.485,1.771,0.485c0.636,0,1.238-0.16,1.808-0.481c0.569-0.32,1.007-0.754,1.313-1.301s0.46-1.164,0.46-1.85c0-1.045-0.344-1.918-1.032-2.619C71.741,94.723,70.894,94.372,69.888,94.372z"/>
                                				<path fill="#231F20" d="M77.531,89.766c0.265,0,0.49,0.094,0.678,0.281c0.188,0.188,0.281,0.416,0.281,0.682c0,0.26-0.094,0.484-0.281,0.672s-0.413,0.282-0.678,0.282c-0.259,0-0.482-0.095-0.669-0.282c-0.188-0.188-0.281-0.412-0.281-0.672c0-0.266,0.094-0.493,0.281-0.682C77.049,89.859,77.272,89.766,77.531,89.766z M76.955,93.484h1.162v9.031h-1.162V93.484z"/>
                                				<path fill="#231F20" d="M84.774,93.252c1.391,0,2.544,0.503,3.459,1.51c0.831,0.919,1.247,2.006,1.247,3.263c0,1.262-0.439,2.364-1.318,3.308c-0.878,0.943-2.008,1.416-3.388,1.416c-1.387,0-2.519-0.473-3.397-1.416s-1.317-2.046-1.317-3.308c0-1.251,0.415-2.335,1.247-3.254C82.221,93.758,83.377,93.252,84.774,93.252z M84.77,94.389c-0.966,0-1.796,0.357-2.49,1.07c-0.694,0.715-1.042,1.578-1.042,2.59c0,0.653,0.159,1.262,0.476,1.826s0.744,1,1.282,1.309c0.539,0.307,1.131,0.46,1.774,0.46c0.645,0,1.235-0.153,1.774-0.46c0.539-0.309,0.967-0.744,1.283-1.309s0.475-1.173,0.475-1.826c0-1.012-0.349-1.875-1.046-2.59C86.56,94.746,85.73,94.389,84.77,94.389z"/>
                                				<path fill="#231F20" d="M105.698,98l0.827,0.875c-0.531,0.471-1.035,0.877-1.513,1.219c0.332,0.306,0.766,0.726,1.3,1.26c0.422,0.42,0.784,0.807,1.087,1.162h-1.702l-1.619-1.67c-1.018,0.82-1.844,1.354-2.48,1.604c-0.637,0.248-1.309,0.373-2.017,0.373c-0.941,0-1.696-0.27-2.267-0.81s-0.854-1.214-0.854-2.022c0-0.608,0.191-1.219,0.576-1.83c0.384-0.611,1.168-1.386,2.352-2.321c-0.669-0.808-1.099-1.403-1.289-1.788s-0.286-0.75-0.286-1.1c0-0.607,0.235-1.111,0.706-1.51c0.542-0.465,1.217-0.697,2.025-0.697c0.509,0,0.974,0.104,1.395,0.308c0.42,0.204,0.741,0.479,0.963,0.825c0.221,0.347,0.332,0.713,0.332,1.101c0,0.409-0.145,0.845-0.433,1.308c-0.288,0.462-0.856,1.053-1.704,1.771L102.948,98l1.213,1.229C104.812,98.75,105.325,98.341,105.698,98z M100.192,96.76c-1.037,0.781-1.717,1.402-2.038,1.865c-0.322,0.462-0.483,0.89-0.483,1.283c0,0.465,0.197,0.876,0.591,1.232c0.394,0.357,0.876,0.536,1.447,0.536c0.433,0,0.856-0.089,1.272-0.266c0.654-0.282,1.404-0.761,2.25-1.438l-1.927-1.984C100.973,97.641,100.603,97.23,100.192,96.76z M100.342,95.145c0.593-0.46,1.039-0.91,1.338-1.354c0.211-0.315,0.316-0.595,0.316-0.838c0-0.283-0.138-0.526-0.412-0.73c-0.274-0.205-0.64-0.308-1.095-0.308c-0.433,0-0.785,0.104-1.057,0.315c-0.272,0.211-0.408,0.459-0.408,0.747c0,0.222,0.056,0.429,0.167,0.622C99.463,94.07,99.847,94.585,100.342,95.145z"/>
                                				<path fill="#231F20" d="M118.314,94.472l-0.748,0.772c-0.621-0.604-1.229-0.905-1.822-0.905c-0.377,0-0.7,0.124-0.97,0.374c-0.269,0.248-0.403,0.539-0.403,0.871c0,0.293,0.111,0.572,0.333,0.838c0.222,0.271,0.688,0.59,1.397,0.955c0.865,0.448,1.452,0.88,1.764,1.295c0.305,0.42,0.457,0.894,0.457,1.42c0,0.741-0.261,1.369-0.781,1.884c-0.521,0.515-1.173,0.772-1.954,0.772c-0.521,0-1.019-0.113-1.492-0.341s-0.866-0.54-1.177-0.938l0.73-0.83c0.594,0.67,1.224,1.005,1.89,1.005c0.466,0,0.862-0.149,1.189-0.448s0.491-0.65,0.491-1.055c0-0.332-0.108-0.627-0.325-0.888c-0.216-0.255-0.704-0.575-1.464-0.963c-0.815-0.421-1.37-0.835-1.664-1.245s-0.44-0.877-0.44-1.402c0-0.687,0.234-1.257,0.702-1.711c0.469-0.453,1.061-0.68,1.775-0.68C116.635,93.252,117.472,93.658,118.314,94.472z"/>
                                				<path fill="#231F20" d="M129.104,95.359l-0.922,0.573c-0.796-1.058-1.882-1.585-3.258-1.585c-1.101,0-2.014,0.354-2.74,1.061c-0.727,0.708-1.09,1.567-1.09,2.578c0,0.658,0.167,1.277,0.501,1.857c0.334,0.581,0.793,1.031,1.376,1.352c0.584,0.321,1.237,0.481,1.961,0.481c1.327,0,2.41-0.528,3.25-1.585l0.922,0.605c-0.432,0.65-1.012,1.154-1.739,1.513c-0.728,0.358-1.556,0.538-2.485,0.538c-1.428,0-2.612-0.454-3.553-1.361c-0.941-0.908-1.412-2.012-1.412-3.312c0-0.875,0.221-1.687,0.66-2.436c0.44-0.75,1.045-1.336,1.814-1.756c0.769-0.421,1.629-0.631,2.581-0.631c0.598,0,1.175,0.091,1.73,0.273c0.557,0.183,1.028,0.42,1.415,0.714C128.504,94.533,128.833,94.906,129.104,95.359z"/>
                                				<path fill="#231F20" d="M131.303,89.998h1.162v5.104c0.471-0.62,0.988-1.083,1.553-1.39c0.564-0.309,1.176-0.461,1.834-0.461c0.675,0,1.274,0.171,1.797,0.514c0.523,0.344,0.909,0.805,1.158,1.383s0.373,1.484,0.373,2.719v4.648h-1.162v-4.309c0-1.041-0.041-1.734-0.124-2.084c-0.144-0.598-0.405-1.047-0.784-1.348c-0.379-0.303-0.876-0.453-1.49-0.453c-0.703,0-1.332,0.232-1.889,0.697c-0.556,0.465-0.923,1.041-1.1,1.727c-0.11,0.443-0.166,1.262-0.166,2.457v3.312h-1.162V89.998z"/>
                                				<path fill="#231F20" d="M145.912,93.252c1.391,0,2.544,0.503,3.459,1.51c0.831,0.919,1.247,2.006,1.247,3.263c0,1.262-0.439,2.364-1.318,3.308c-0.878,0.943-2.008,1.416-3.388,1.416c-1.387,0-2.519-0.473-3.397-1.416s-1.317-2.046-1.317-3.308c0-1.251,0.415-2.335,1.247-3.254C143.358,93.758,144.515,93.252,145.912,93.252z M145.907,94.389c-0.966,0-1.796,0.357-2.49,1.07c-0.694,0.715-1.042,1.578-1.042,2.59c0,0.653,0.159,1.262,0.476,1.826s0.744,1,1.282,1.309c0.539,0.307,1.131,0.46,1.774,0.46c0.645,0,1.235-0.153,1.774-0.46c0.539-0.309,0.967-0.744,1.283-1.309s0.475-1.173,0.475-1.826c0-1.012-0.349-1.875-1.046-2.59C147.697,94.746,146.868,94.389,145.907,94.389z"/>
                                				<path fill="#231F20" d="M157.01,93.252c1.391,0,2.544,0.503,3.459,1.51c0.831,0.919,1.247,2.006,1.247,3.263c0,1.262-0.439,2.364-1.318,3.308c-0.878,0.943-2.008,1.416-3.388,1.416c-1.387,0-2.519-0.473-3.397-1.416s-1.317-2.046-1.317-3.308c0-1.251,0.415-2.335,1.247-3.254C154.456,93.758,155.612,93.252,157.01,93.252z M157.005,94.389c-0.966,0-1.796,0.357-2.49,1.07c-0.694,0.715-1.042,1.578-1.042,2.59c0,0.653,0.159,1.262,0.476,1.826s0.744,1,1.282,1.309c0.539,0.307,1.131,0.46,1.774,0.46c0.645,0,1.235-0.153,1.774-0.46c0.539-0.309,0.967-0.744,1.283-1.309s0.475-1.173,0.475-1.826c0-1.012-0.349-1.875-1.046-2.59C158.795,94.746,157.966,94.389,157.005,94.389z"/>
                                				<path fill="#231F20" d="M163.657,89.998h1.162v12.518h-1.162V89.998z"/>
                                			</g>
                                		</g>
                                	</svg>
                                </a>
                                <p><strong>Brought to you by <a href="<?php echo $link; ?>" target="_blank" >MIGHTYminnow</a></strong></p>
                                <p>
                                	We make custom websites and plugins.<br />
                                	Need help with a project?
                                </p>
                                <a class="button-primary" href="<?php echo $link; ?>" target="_blank">Contact Us</a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <br class="clear" />
            <!-- Full column stuff goes here -->
        </div>
        <?php
	}

	function output_add_button() {
		printf(
			'<button type="button" class="button add-group">%s</button>',
			__( '+ Add More', 'equal-height-columns' )
		);
	}

	/**
	 * Register the plugin settings.
	 *
	 * @since    1.0.0
	 */
	function register_settings() {

		register_setting(
            $this->plugin_name, // Option group
            $this->plugin_name, // Option name
            array( $this, 'validate' ) // Validate
        );

        add_settings_section(
            'main-settings', // ID
            __( '<br />Instructions', 'equal-height-columns' ), // Title
            array( $this, 'output_element_section_info' ), // Callback
            $this->plugin_name // Page
        );

        // Set number of fields to output (number of saved groups, or 1 if none are saved).
	$field_count = ( ( $this->options ) &&  count( $this->options[ $this->options_group_slug ] ) ) ? count( $this->options[ $this->options_group_slug ] ) : 1;

        // Output correct number of fields.
        for ( $i = 1; $i <= $field_count; $i++ ) {

	        add_settings_field(
	            'elements-group-' . $i, // ID
	            __( 'Group #', 'equal-height-columns' ) . '<span class="index-number">' . $i . '<span>', // Title
	            array( $this, 'output_elements_fields' ), // Callback
	            $this->plugin_name, // Page
	            'main-settings', // Section
	            array( // Args
	                'id' => $this->options_group_slug,
	                'index' => $i,
	            )
	        );

	    }

	}

	function validate( $input ) {

		// Validate input fields.
		foreach( $input[ $this->options_group_slug ] as $key => $group ) {

			// Validate selector.
			$selector = $group['selector'];

			$input[ $this->options_group_slug ][ $key ]['selector'] = wp_strip_all_tags( $selector, true );

			// Validate breakpoint.
			$breakpoint = $group['breakpoint'];
			$input[ $this->options_group_slug ][ $key ]['breakpoint'] = intval( $breakpoint );

		}

		return $input;
	}

	function output_element_section_info() {
		?>
		<p><?php _e( '<b>Selector:</b> jQuery/CSS selectors of the elements you wish to target. Examples: <code>.col-1, .col-2</code> or <code>.container .col</code>', 'equal-height-columns' ); ?></p>
		<p><?php _e( '<b>Breakpoint:</b> Screen width <i>above</i> which you want equal heights to take effect.</code>', 'equal-height-columns' ); ?></p>
		<br />
		<?php
	}

	function output_elements_fields( $args ) {

		// Selector input
		$field_id = 'selector-' . $args['index'];
		printf(
            '<div class="ehc-input-wrapper"><label for="%s">%s</label> <input type="text" id="%s" name="%s[%s][%s][%s]" data-index="%s" value="%s" placeholder="%s" /></div>',
            $field_id,
            __( 'Selector:', 'equal-height-columns' ),
            $field_id,
            $this->plugin_name,
            $args['id'],
            $args['index'],
            'selector',
            $args['index'],
            isset( $this->options[ $args['id'] ][ $args['index'] ]['selector'] ) ? esc_attr( $this->options[ $args['id'] ][ $args['index'] ]['selector'] ) : '',
            __( '.col-1, .col-2, .col-3', 'equal-height-columns' )
        );

		// Breakpoint input
		$field_id = 'breakpoint-' . $args['index'];
        printf(
            '<div class="ehc-input-wrapper"><label for="%s">%s</label> <input type="number" id="%s" name="%s[%s][%s][%s]" data-index="%s" value="%s" /> px</div>',
            $field_id,
            __( 'Breakpoint:', 'equal-height-columns' ),
            $field_id,
            $this->plugin_name,
            $args['id'],
            $args['index'],
            'breakpoint',
            $args['index'],
            isset( $this->options[ $args['id'] ][ $args['index'] ]['breakpoint'] ) ? esc_attr( $this->options[ $args['id'] ][ $args['index'] ]['breakpoint'] ) : '768'
        );

        // Remove button
        printf(
            '<button type="button" class="button remove-group">%s</button>',
            __( 'Remove', 'equal-height-columns' )
        );

	}
}
